(* ::Subsubsection:: *)
(*Triplet package*)

(* ::Text:: *)
(*This impements the Triplet Telescope optical system*)

(* ::Text:: *)
(*Chapter*)

(* ::Text:: *)
(*This file should be in the following path*)

(* ::Text:: *)
(*FileNameJoin[{$UserBaseDirectory, "Applications/GeometricOptics/Triplet"}]*)

BeginPackage["GeometricOptics`Packages`Triplet`", {"GeometricOptics`"}]

Options[Triplet] = {OutputType->"Report", OutputLevel->"Full"};
(* 	OutputType can be 
					"Report", generates a new report (notebook) for each call to the function
					"Print", prints results in the current position of the evaluating notebook
					"Basic", gives the list of results
					"Values" or any other directive, gives the list of values (no names of variables)
	OutputLevel can be
					"Minimal", gives only output values, those calculated by the function
					"Full", gives the list of input parameters and output values
*)
Triplet::usage="Triplet calculates parameters for a Triplet Telescope optical system.";

TripletInputPanel::usage = "TripletInputPanel gives the GUI panel to work with Triplet";

$TripletInputVariables = {"thick", "ind", "diam", "theta", "waves", "focal"};
{thick, ind, diam, theta, waves, focal};

Begin["`Private`"]

Triplet[{thick_, ind_, diam_, theta_, waves_, focal_}, opts___]:= Triplet[thick, ind, diam, theta, waves, focal];

Triplet[thick_, ind_, diam_, theta_, waves_, focal_, OptionsPattern[]] := 
Quiet@Module[{em1, em2, em3, em4, sol0, em5, s1, sol2, sol1, sol3, rad, c1, c2, c3, c4, c5, fun, gr, outputs, inPanel, outPanel},
 
		If[ArgumentsQ["Triplet", {$TripletInputVariables, {thick, ind, diam, theta, waves, focal}}],
			AppendTo[$ExamplesStack, 
					 <|"PackageID" -> 13, "PackageName" -> "Triplet", "Arguments" -> <|"thick" -> thick, "ind" -> ind, "diam" -> diam, "theta" -> theta,
																						  "waves" -> waves, "focal" -> focal|>|>];
			$ExamplesStack = DeleteExampleDuplicates[$ExamplesStack];
			
			TotalAberrations[{1/c1, 1/c2, 1/c3, 1/c4, 1/c5}, {0, 0, 0, 0}, ind, {0, 0, 0, 0, 0}, diam/2, 0, 0, -Infinity, y1, theta, waves];
			
			(* Focal length in the variables c1, c2, c3, c4 *)
			em1 = Simplify[Numerator[Together[(1/GOfocalLength[[1]]) - (1/focal)]]];
			
			(*Back focals with respect to green, blue, red*)
			em2 = Simplify[Numerator[Together[1/GOdistancegauss[[2, GOn]] - 1/GOdistancegauss[[1, GOn]]]]];
			em3 = Simplify[Numerator[Together[1/GOdistancegauss[[3, GOn]] - 1/GOdistancegauss[[1, GOn]]]]];
			
			sol0 = Flatten@Solve[{em1 == 0, em2 == 0, em3 == 0}, {c1, c3, c5}];
			
			em4 = Simplify[GOSphericalCoefficient /. sol0];
			em5 = Simplify[GOComaCoefficient /. sol0];
			
			s1 = Solve[{em4 == 0, em5 == 0}, {c2, c4}];
			sol2 = First@First[SortBy[Map[{#, Total[ReplaceAll[{c2, c4}, #]^2]} &, s1], Last]];
			sol1 = sol0 /. sol2;
			sol3 = Sort[Join[sol1, sol2]];
			
			rad = {1/ReplaceAll[c1, sol3], 1/ReplaceAll[c2, sol3], 1/ReplaceAll[c3, sol3], 1/ReplaceAll[c4, sol3], ReplaceAll[c5, sol3]};
			TotalAberrations[rad, thick, ind, {0, 0, 0, 0, 0}, diam/2, 0, 0, -Infinity, y1, theta, waves];
								
			(* Drawing *)
			fun = MapThread[If[#1 > 0, #1 + #2 - Sqrt[#^2 - x^2], #1 + #2 + Sqrt[#^2 - x^2]] &, {rad, Prepend[Accumulate[thick], 0]}];
			gr = Plot[Evaluate@fun, {x, -diam/2, diam/2}, Filling -> {1 -> {2}, 3 -> {4}}, PlotRange -> All, AspectRatio->1/5];
			
			outputs = {{"Radii of triplet", Row[Riffle[rad, ", "]]},
						{"Focal length in green-light", GOfocalLength[[1]]},
						{"Height of the image", GOimageHeight[[1, GOn]]},
						{"Spherical total aberration", GOaberration[[1]]},
						{"Total sagittal coma", GOcoma[[1]]},
						{"Total Astigmatism", GOastigmatism[[1]]},
						{"Blue-red back focals",   (GOdistancegauss[[2, GOn]] - GOdistancegauss[[3, GOn]])},
						{"Blue-green back focals", (GOdistancegauss[[2, GOn]] - GOdistancegauss[[1, GOn]])},
						{"Red-green back focals",  (GOdistancegauss[[3, GOn]] - GOdistancegauss[[1, GOn]])},
						{"Residual chromatic aberration", ToString[100*(GOdistancegauss[[2, GOn]] - GOdistancegauss[[3, GOn]])/focal]<>"%"},
						{"Spherical aberration in blue light", GOaberration[[2]]},
						{"Spherical aberration in red light", GOaberration[[3]]},
						{"Coma in blue light", GOcoma[[2]]},
						{"Coma in red light", GOcoma[[3]]},
						{"Plots", gr}};
			
			(* defines the two panels, input parameters and output values *)
			inPanel = Grid[{{"Thickness of lenses", "thick", Grid[{thick}]},
							{"List of refractive indices", "ind", Grid[ind]},
							{"Diameter of primary mirror", "diam", diam},
							{"Field angle  in degree", "theta", theta},
							{"Wavelengths of the chosen refractive indices", "waves", Grid[{waves}]},
							{"Focal", "focal", focal}},
							Alignment -> {{Left, Left, Right}, Center}, 
							Spacings -> {2, 1}, 
							Dividers -> Center, 
							FrameStyle -> LightGray,
							BaseStyle->{"InputParameterBottom"}];
  
			outPanel = Grid[outputs, 
							Alignment -> {{Left, Right}, Center}, 
							Spacings -> {2, 1}, 
							Dividers -> Center, 
							FrameStyle -> LightGray,
							BaseStyle->{"OutputValueBottom"}];
				
			(* generates the type of output required *)
			Switch[OptionValue[OutputType],
					"Report",
					GenerateDocument[TemplateApply[$ReportTemplate, 
										Join[<|	"title" -> $GeometricOpticsPackagesList[SelectFirst[#PackageName == "Triplet" &], "Description"], 
												"date" -> DateString[], 
												"function" -> "Triplet", 
												"outputlevel" -> OptionValue[OutputLevel],
												"inPanel" -> inPanel, 
												"outPanel" -> outPanel |>]]];,

					"Print",
					CellPrint[TextCell[TemplateApply[$PrintTemplate, 
											Join[<|	"title" -> $GeometricOpticsPackagesList[SelectFirst[#PackageName == "Triplet" &], "Description"], 
													"date" -> DateString[], 
													"function" -> "Triplet", 
													"outputlevel" -> OptionValue[OutputLevel],
													"inPanel" -> inPanel, 
													"outPanel" -> outPanel |>]], "Text"]];,
					"Basic",
					CellPrint[TextCell[
					TemplateApply[$BasicTemplate, 
									Join[<| "outputlevel" -> OptionValue[OutputLevel],
											"inputs" -> {{"thick", thick},
														 {"ind", ind},
														 {"diam", diam},
														 {"theta", theta},
														 {"waves", waves},
														 {"focal", focal}},
											"outputs" -> outputs |>]], "Output"]],
					_,
					CellPrint[TextCell[DeleteCases[outputs[[All, 2]], Alternatives["", Style[___]]], "Output"]]],

							
			(* Arguments are not correct *)
			MessageDialog["Triplet not executed, the number or the type of the arguments may be incorrect.", WindowTitle->"Warning: example not generated"];]];

TripletInputPanel[]:=
DynamicModule[{outputtype = "Report", package = "Triplet"},
	examplesAll = Join[	Cases[Values[$ExamplesStack], {_, package, arguments_} :> arguments], 
						Cases[Normal@Values[$ExamplesArchive], {_, _, package, arguments_, _} :> arguments]];
    examples = Map[Framed[Grid[Transpose[KeyValueMap[List, Association[#]]], Alignment -> Center, Spacings -> {1, 1}, Dividers -> Center], FrameStyle -> LightGray] &, examplesAll];
	example = "Browse...";
	Panel[Column[{	DynamicWrapper[Style[NameFromPackageName[package] <> " Input Panel", "Subsection"], 
									If[NumericQ[example], {thick, ind, diam, theta, waves, focal} = ReplaceAll[$TripletInputVariables, examplesAll[[example]]]]],
					Style["Insert values for each argument, then use Evaluate to run "<>package<>" function", "Text"],
					Grid[{{"Thickness of lenses" , "thick", Tooltip[InputField[Dynamic[thick], Alignment -> Center], "Insert the list of thickness of lenses"]},
						  {"List of refractive indices", "ind", Tooltip[InputField[Dynamic[ind], Alignment -> Center, FieldSize->30], "Insert the list of refractive indices"]},
						  {"Diameter of the primary mirror", "diam", Tooltip[InputField[Dynamic[diam], Alignment -> Center], "Input the value of diam"]},
						  {"Field angle in degrees", "\[Theta]", Tooltip[InputField[Dynamic[theta], Alignment -> Center], "Insert the value of \[Theta]"]},
						  {"Wavelengths of the choose refractive indices", "waves", Tooltip[InputField[Dynamic[waves], Alignment -> Center], "Input the list of wavelengths"]},
						  {"Focal", "focal", Tooltip[InputField[Dynamic[diam], Alignment -> Center], "Input the value of focal"]}},
						Spacings -> {1, 0},
						Alignment -> {{Left, Left, Right}, Center}, 
						Dividers -> Center, 
						FrameStyle -> LightGray],
					OpenerView[{"Load an example from the archives (current session and saved DB)",
								Row[{Dynamic@PopupMenu[Dynamic[example], Thread[Rule[Range[Length[examples]], examples]], If[examples === {}, "No example saved", "Browse..."], FrameMargins -> 3, Alignment -> Center],
									 Spacer[5],
									 Button["Update list", (examplesAll = Join[	Cases[Values[$ExamplesStack], {_, package, arguments_} :> arguments], 
																				Cases[Normal@Values[$ExamplesArchive], {_, _, package, arguments_, _} :> arguments]];
															examples = Map[Framed[Grid[Transpose[KeyValueMap[List, Association[#]]], 
																						Alignment -> Center, 
																						Spacings -> {1, 1}, 
																						Dividers -> Center], 
																						FrameStyle -> LightGray]&, examplesAll];
															example = "Browse..."), Method -> "Queued"]}]}, 
								Alignment -> Center, 
								Spacings -> 1.5],
					Row[{"Define the type of output to generate", 
						 Spacer[5],
						 RadioButtonBar[Dynamic[outputtype], {"Report" -> Tooltip["Report", "Generates a new notebook reporting a summary of the calculation"], 
															  "Print" -> Tooltip["Print", "Print the table of the calculation done inside the current notebook"], 
															  "Basic" -> Tooltip["Basic", "Generate a list of computed output with label"], 
															  "Values" -> Tooltip["Values", "Return only the list of output values"]}]}],
					Row[{Button["Evaluate", ToExpression[package][Apply[Sequence, {thick, ind, diam, theta, waves, focal}], OutputType -> outputtype], Method -> "Queued"],
						 Button["Clear all", Map[Clear, Unevaluated[{thick, ind, diam, theta, waves, focal}]]]}]}, 
				Spacings -> 2, 
				Alignment -> Center],
	BaseStyle -> {InputFieldBoxOptions -> {FieldSize -> {15, 1}}}]];
			
  
  End[]
  EndPackage[]